using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Diagnostics;

namespace GlassMoth
{
  public partial class Form1 : Form
  {
    // margins for API call used in Paint
    private VistaApi.Margins marg;

    // Rectangles; one for each edge of the form
    private Rectangle topRect = Rectangle.Empty;
    private Rectangle botRect = Rectangle.Empty;
    private Rectangle lefRect = Rectangle.Empty;
    private Rectangle rigRect = Rectangle.Empty;

    public Form1()
    {
      InitializeComponent();

      trackBar1.Minimum = 0;
      trackBar1.Maximum = this.ClientSize.Height;

      this.FitGlass();
    }

    private void FitGlass()
    {
      // If DWM is not enabled then get out
      if (!this.IsGlassEnabled())
      {
        return;
      }

      // Set the Margins to their default values
      marg.Top = trackBar1.Value; // extend from the top
      marg.Left = 0;  // not used in this sample but could be
      marg.Right = 0; // not used in this sample but could be
      marg.Bottom = 0;// not used in this sample but could be

      this.Paint += new PaintEventHandler(this.Form1_Paint);

      // call the function that gives us glass, 
      // passing a reference to our inset Margins
      VistaApi.DwmExtendFrameIntoClientArea(this.Handle, ref marg);      
    }

    private bool IsGlassEnabled()
    {
      if (Environment.OSVersion.Version.Major < 6)
      {
        Debug.WriteLine("How about trying this on Vista?");
        return false;
      }

      //Check if DWM is enabled
      bool isGlassSupported = false;
      VistaApi.DwmIsCompositionEnabled(ref isGlassSupported);
      return isGlassSupported;
    }

    // Alpha-blending Paint after the glass extension
    // this seems better than the winforms transparency approach because here we can click on the glass!
    private void Form1_Paint(object sender, PaintEventArgs e)
    {
      // black brush for Alpha transparency
      SolidBrush blackBrush = new SolidBrush(Color.Black);

      Graphics g = e.Graphics;

      if (this.IsGlassEnabled())
      {      
        // setup the rectangles
        topRect = new Rectangle(0, 0, this.ClientSize.Width, marg.Top);
        lefRect = new Rectangle(0, 0, marg.Left, this.ClientSize.Height);
        rigRect = new Rectangle(this.ClientSize.Width - marg.Right, 0, marg.Right, this.ClientSize.Height);
        botRect = new Rectangle(0, this.ClientSize.Height - marg.Bottom, this.ClientSize.Width, marg.Bottom);

        // Fill Rectangles
        g.FillRectangle(blackBrush, topRect);
        g.FillRectangle(blackBrush, lefRect);
        g.FillRectangle(blackBrush, rigRect);
        g.FillRectangle(blackBrush, botRect);
      }

      g.DrawString("hello. Am I on glass?", this.Font, blackBrush, 10, 10);

      blackBrush.Dispose();
    }

    private void button1_Click(object sender, EventArgs e)
    {
      this.Paint -= new System.Windows.Forms.PaintEventHandler(this.Form1_Paint);
      this.RecreateHandle(); //needed if changing on the fly
      this.FitGlass();
    }
        
    protected override void WndProc(ref Message m)
    {      
      base.WndProc(ref m);

      if (m.Msg == VistaApi.WM_NCHITTEST // if this is a click
        && m.Result.ToInt32() == VistaApi.HTCLIENT // ...and it is on the client
        && this.IsOnGlass(m.LParam.ToInt32())) // ...and specifically in the glass area
      {
        m.Result = new IntPtr(VistaApi.HTCAPTION); // lie and say they clicked on the title bar
      }
    }

    private bool IsOnGlass(int lParam)
    {
      // sanity check
      if (!this.IsGlassEnabled())
      {
        return false;
      }

      // get screen coordinates
      int x = (lParam << 16) >> 16; // lo order word
      int y =  lParam        >> 16; // hi order word

      // translate screen coordinates to client area
      Point p = this.PointToClient(new Point(x, y));

      // work out if point clicked is on glass
      if (topRect.Contains(p) || lefRect.Contains(p) || rigRect.Contains(p) || botRect.Contains(p))
      {
        return true;
      }

      return false;
    }
  }
}